<!doctype html>
<html lang="en">

<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Custom Spreadsheet</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        th,
        td {
            min-width: 120px;
            resize: horizontal;
            overflow: auto;
        }

        th.serial-col,
        td.serial-col {
            min-width: 60px;
            resize: none;
        }

        /* Style select dropdowns */
        select {
            padding: 4px 6px;
            border-radius: 0.375rem;
            border: 1px solid #d1d5db;
            font-size: 0.875rem;
            outline: none;
        }

        select:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 1px #2563eb;
        }
    </style>
</head>

<body class="bg-gray-50 p-6">
    <div class="w-full">
        <div class="flex items-center justify-between mb-4">
            <h1 class="text-2xl font-semibold">Custom Data Table</h1>
            <div class="space-x-2">
                <button id="add-row" class="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700">Add
                    Row</button>
                <button id="upload-data" class="px-3 py-1 bg-green-600 text-white rounded hover:bg-green-700">Upload
                    Data</button>
                <button id="extract-data" class="px-3 py-1 bg-indigo-600 text-white rounded hover:bg-indigo-700">Extract
                    Data</button>
            </div>
        </div>

        <div class="overflow-auto border rounded bg-white w-full">
            <table id="sheet" class="w-full border-collapse table-auto">
                <thead class="bg-gray-100">
                    <tr class="uppercase">
                        <th class="p-2 border serial-col">S.No</th>
                        <th class="p-2 border">Date</th>
                        <th class="p-2 border">Country</th>
                        <th class="p-2 border">Customer Name</th>
                        <th class="p-2 border">Dialing Status</th>
                        <th class="p-2 border">Comments</th>
                        <th class="p-2 border">Time & Date</th>
                        <th class="p-2 border">Email</th>
                        <th class="p-2 border">Number</th>
                        <th class="p-2 border">Call Type</th>
                        <th class="p-2 border">Company</th>
                        <th class="p-2 border">Last Amount</th>
                        <th class="p-2 border">BTTR</th>
                        <th class="p-2 border">Payment Dates</th>
                        <th class="p-2 border">Final Status</th>
                        <th class="p-2 border">Agent Name</th>
                        <th class="p-2 border">Delete</th>
                    </tr>
                </thead>
                <tbody id="sheet-body"></tbody>
            </table>
        </div>

        <p class="text-sm text-gray-600 mt-3">
            Editable fields: <b>Dialing Status</b>, <b>Comments</b>, <b>Time & Date</b>, and <b>Final Status</b>.
            Use the delete button to remove a row. Resize columns by dragging.
        </p>
    </div>

    <script>
        const tbody = document.getElementById('sheet-body');
        let rowCount = 0;

        // Dropdown options for statuses
        const statuses = [{
                text: "Sale Done",
                color: "bg-green-500 text-white"
            },
            {
                text: "Denied",
                color: "bg-red-500 text-white"
            },
            {
                text: "Call Back",
                color: "bg-blue-400 text-white"
            },
            {
                text: "Pitched",
                color: "bg-blue-600 text-white"
            },
            {
                text: "Conversation",
                color: "bg-yellow-400 text-black"
            },
            {
                text: "Voicemail Left",
                color: "bg-green-600 text-white"
            },
            {
                text: "Customer Services",
                color: "bg-pink-400 text-white"
            },
            {
                text: "Hung Up",
                color: "bg-red-700 text-white"
            },
            {
                text: "Not Available",
                color: "bg-gray-400 text-white"
            },
            {
                text: "Voicemail",
                color: "bg-indigo-500 text-white"
            },
            {
                text: "Not In Service",
                color: "bg-gray-700 text-white"
            }
        ];

        function createStatusDropdown(selected = "") {
            const select = document.createElement('select');
            statuses.forEach(s => {
                const option = document.createElement('option');
                option.value = s.text;
                option.textContent = s.text;
                option.className = s.color;
                if (s.text === selected) option.selected = true;
                select.appendChild(option);
            });
            return select;
        }

        function createRow(data = {}) {
            rowCount++;
            const tr = document.createElement('tr');
            const fields = [
                'S.No', 'Date', 'Country', 'Customer Name', 'Dialing Status', 'Comments',
                'Time & Date', 'Email', 'Number', 'Call Type', 'Company',
                'Last Amount', 'BTTR', 'Payment Dates', 'Final Status',
                'Agent Name', 'Delete'
            ];

            fields.forEach((field) => {
                const td = document.createElement('td');
                td.className = 'border p-2 text-sm';

                if (field === 'S.No') {
                    td.classList.add('serial-col');
                    td.textContent = rowCount;
                } else if (field === 'Dialing Status' || field === 'Final Status') {
                    td.appendChild(createStatusDropdown(data[field] || ""));
                } else if (['Comments', 'Time & Date'].includes(field)) {
                    const div = document.createElement('div');
                    div.contentEditable = 'true';
                    div.className = 'min-h-[28px] px-2 focus:outline-none';
                    div.textContent = data[field] || '';
                    td.appendChild(div);
                } else if (field === 'Delete') {
                    const btn = document.createElement('button');
                    btn.textContent = 'Delete';
                    btn.className = 'px-2 py-1 bg-red-600 text-white rounded hover:bg-red-700 text-xs';
                    btn.addEventListener('click', () => {
                        tr.remove();
                        updateSerialNumbers();
                    });
                    td.appendChild(btn);
                } else {
                    td.textContent = data[field] || '';
                }

                tr.appendChild(td);
            });

            tbody.appendChild(tr);
        }

        function updateSerialNumbers() {
            let sn = 1;
            tbody.querySelectorAll('tr').forEach(row => {
                row.querySelector('td').textContent = sn++;
            });
            rowCount = sn - 1;
        }

        document.getElementById('add-row').addEventListener('click', () => createRow());

        document.getElementById('upload-data').addEventListener('click', () => {
            alert('Upload Data clicked (implement upload logic)');
        });

        document.getElementById('extract-data').addEventListener('click', () => {
            const rows = [];
            tbody.querySelectorAll('tr').forEach(tr => {
                const cells = Array.from(tr.querySelectorAll('td')).map(td => td.innerText.trim());
                rows.push(cells);
            });
            console.log(rows);
            alert('Data extracted. Check console.');
        });

        // Add some dummy rows
        createRow({
            Date: '2025-09-20',
            Country: 'India',
            'Customer Name': 'John Doe',
            'Dialing Status': 'Call Back',
            Comments: 'Called twice',
            'Time & Date': '10:30 AM',
            Email: 'john@example.com',
            Number: '9876543210',
            'Call Type': 'Follow-up',
            Company: 'ABC Pvt Ltd',
            'Last Amount': '$1200',
            BTTR: 'Yes',
            'Payment Dates': '2025-09-25',
            'Final Status': 'Sale Done',
            'Agent Name': 'Kanhaiya'
        });

        createRow({
            Date: '2025-09-19',
            Country: 'USA',
            'Customer Name': 'Jane Smith',
            'Dialing Status': 'Voicemail',
            Comments: 'Positive response',
            'Time & Date': '4:45 PM',
            Email: 'jane@example.com',
            Number: '1234567890',
            'Call Type': 'Inquiry',
            Company: 'XYZ Corp',
            'Last Amount': '$800',
            BTTR: 'No',
            'Payment Dates': '2025-09-21',
            'Final Status': 'Denied',
            'Agent Name': 'Ravi'
        });
    </script>
</body>

</html>
